/*
 * Project          C++ Implementation "Unit Conversion Table"
 * Author           Amar Khelil, www.khelil.de, Copyright 2014-2015
 * Documentation    http://www.khelil.de/Ref_Projekte_Intern/C/Project_C++_ConversionTable_EN.html
 */

//C++ Standard Library
#include <iostream>
#include <iomanip>

#include "global_UnitConversionTable.h"
#include "converter_time.h"

//=====================================
//### CONSTRUCTORS DESTRUCTORS
//=====================================

//-------------------------------------------------------------------------------------------------
converter_time::converter_time(void): converter_generic()
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Constructor converter_time::converter_time() is being invoked";
    }

    init();

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Constructor converter_time::converter_time() [End]";
    }
}

//-------------------------------------------------------------------------------------------------
converter_time::converter_time(QString CName): converter_generic(CName)
  //-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Constructor converter_time::converter_time("
                  << CName.toLocal8Bit().data()
                  << ") [Begin]";
    }

    init();

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Constructor converter_time::converter_time("
                  << CName.toLocal8Bit().data()
                  << ") [End]";
    }
}

//-------------------------------------------------------------------------------------------------
converter_time::~converter_time()
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Destructor converter_time::~converter_time"
                  << "(name="
                  << name_converter.toLocal8Bit().data()
                  << ", id="
                  << id_Conv
                  << ") [Begin]";
    }
}

//=====================================
//### INITIALIZE OBJECT
//=====================================

//  basic initialization for a specific converter associated with time
//-------------------------------------------------------------------------------------------------
void  converter_time::init(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">converter_time::init() [Begin]";
    }

//  Conversion variables: generic <-> specific
    n_units                     = nUNIT_TIME;
    generateArray_names_unit();        //  defines names_unit[]
    generateArray_vals_unit();          //  defines vals_unit[]
//  id_ref_unit = 0;                    // By default the reference unit is the first one (May be changed by the user)

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl<< ">converter_time::init() [End]";
    }

    return;
}

//### GENERATE Arrays
//-------------------------------------------------------------------------------------------------
void  converter_time::generateArray_names_unit(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
            std::cout << std::endl
                      << ">converter_time::generateArray_names_unit() [Begin] (creates and populates array for unit names)";
    }
    names_unit                  = new QString[nUNIT_TIME];
    names_unit[unit_day]        = "day";
    names_unit[unit_year]       = "year";
    names_unit[unit_month]      = "month";
    names_unit[unit_hour]       = "hour";
    names_unit[unit_second]     = "second";
    return;
}

//-------------------------------------------------------------------------------------------------
void  converter_time::generateArray_vals_unit(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">converter_time::generateArray_vals_unit() [Begin] (creates and populates array of unit values)";
    }

    vals_unit                   = new double[nUNIT_TIME];
    setValue_D(                 glb_UCT::ARBITRARY_VALUE);
    return;
}

//-------------------------------------------------------------------------------------------------
void  converter_time::setVal_UID(double val, int uid)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">converter_time::setVal_UID("
                  << "val="<< val<<", uid="<< uid
                  << ") [Begin] )";
    }

    switch (uid)
    {

        case unit_day:
            setValue_D(val);
        break;

        case unit_year:
            setValue_Y(val);
        break;

        case unit_month:
            setValue_M(val);
        break;

        case unit_hour:
            setValue_H(val);
        break;

        case unit_second:
            setValue_S(val);
        break;

        default:
            std::cout << std::endl
                  << "->ERROR uid="<< uid << " is not a legal unit ID!";
    }
    return;
}


//##
// -2- SET functions
// The consistency between temperature values acoording to all registered units is always maintained

//-------------------------------------------------------------------------------------------------
void converter_time::setValue_Y(double year)
//-------------------------------------------------------------------------------------------------
{
    double  day;
    day                     = convert_Y_2_D(year);

     vals_unit[unit_year]    = year;
     vals_unit[unit_month]   = convert_D_2_M(day);
     vals_unit[unit_day]     = day;
     vals_unit[unit_hour]    = convert_D_2_H(day);
     vals_unit[unit_second]  = convert_D_2_S(day);

    return;
}

//-------------------------------------------------------------------------------------------------
void converter_time::setValue_M(double month)
//-------------------------------------------------------------------------------------------------
{
    double  day;
    day                     = convert_M_2_D(month);


    vals_unit[unit_year]    = convert_D_2_Y(day);
    vals_unit[unit_month]   = month;
    vals_unit[unit_day]     = day;
    vals_unit[unit_hour]    = convert_D_2_H(day);
    vals_unit[unit_second]  = convert_D_2_S(day);

    return;
}

//-------------------------------------------------------------------------------------------------
void converter_time::setValue_D(double day)
//-------------------------------------------------------------------------------------------------
{
    vals_unit[unit_year]    = convert_D_2_Y(day);
    vals_unit[unit_month]   = convert_D_2_M(day);
    vals_unit[unit_day]     = day;
    vals_unit[unit_hour]    = convert_D_2_H(day);
    vals_unit[unit_second]  = convert_D_2_S(day);

    return;
}

//-------------------------------------------------------------------------------------------------
void converter_time::setValue_H(double hour)
//-------------------------------------------------------------------------------------------------
{
    double  day;
    day                     = convert_H_2_D(hour);

    vals_unit[unit_year]    = convert_D_2_Y(day);
    vals_unit[unit_month]   = convert_D_2_M(day);
    vals_unit[unit_day]     = day;
    vals_unit[unit_hour]    = hour;
    vals_unit[unit_second]  = convert_D_2_S(day);

    return;
}

//-------------------------------------------------------------------------------------------------
void converter_time::setValue_S(double second)
//-------------------------------------------------------------------------------------------------
{
    double  day;
    day                     = convert_S_2_D(second);

    vals_unit[unit_year]    = convert_D_2_Y(day);
    vals_unit[unit_month]   = convert_D_2_M(day);
    vals_unit[unit_day]     = day;
    vals_unit[unit_hour]    = convert_D_2_S(day);
    vals_unit[unit_second]  = second;

    return;
}



//### CONVERSION METHODS
//-------------------------------------------------------------------------------------------------
double converter_time::convert_D_2_Y(   double day)     // day      to year
//-------------------------------------------------------------------------------------------------
{
    double year = day/365.25;
    return year;
}

//-------------------------------------------------------------------------------------------------
double converter_time::convert_D_2_M(   double day)
//-------------------------------------------------------------------------------------------------
{
    double month    = day/30.0;
    return month;
}

//-------------------------------------------------------------------------------------------------
double converter_time::convert_D_2_H(   double day)     // day      to hour
//-------------------------------------------------------------------------------------------------
{
    double hour     = day*24.0;
    return hour;
}

//-------------------------------------------------------------------------------------------------
double converter_time::convert_D_2_S(   double day)     // day      to second
//-------------------------------------------------------------------------------------------------
{
    double second   = day*3600.0*24.0;
    return second;
}

//-------------------------------------------------------------------------------------------------
double converter_time::convert_Y_2_D(   double year)     // year     to day
//-------------------------------------------------------------------------------------------------
{
    double day  = year*365.25;
    return day;
}

//-------------------------------------------------------------------------------------------------
double converter_time::convert_M_2_D(   double month)     // month    to day
//-------------------------------------------------------------------------------------------------
{
    double day  = month *30.0;
    return day;
}

//-------------------------------------------------------------------------------------------------
double converter_time::convert_H_2_D(   double hour)     // hour     to day
//-------------------------------------------------------------------------------------------------
{
    double day  = hour / 24.0;
    return day;
}

//-------------------------------------------------------------------------------------------------
double converter_time::convert_S_2_D(   double second)     // second   to day
//-------------------------------------------------------------------------------------------------
{
    double day  = second/3600.0/24.0;
    return day;
}

