/*
 * Project          C++ Implementation "Unit Conversion Table"
 * Author           Amar Khelil, www.khelil.de, Copyright 2014-2016
 * Documentation    http://www.khelil.de/Ref_Projekte_Intern/C/Project_C++_ConversionTable_EN.html
 *
 * FILE             converter_generic.h
 * Last modified    June 2016
 */

#ifndef CONVERTER_GENERIC_H
#define CONVERTER_GENERIC_H

//C++ Standard Library
// Framework qt
#include <QString>
// Framework CL_UnitConversionTable
#include "global_UnitConversionTable.h"

// ABSTRACT CLASS converter_generic: parent of all physical converter classes
//
// BASIC CHARACTERISTICS
//  Abstraction level: set of converters
// -1-  All supported physical converters are listed in enum_converter (Number of converters supported is n_Conv).

// Abstraction level: within a converter
// -1-  A physical converter is associated with several units of which values correspond to the same "state" (Number of units registered is n_units).
// -2-  Each registered unit is associated to a unique identification number uid, equal to its order in the unit sequence.
//      The first unit registered has uid=0, the uid increments for each additional one
// -3-  One unit uid is marked as the "reference unit" UID_red
// -4-  Each unit is associated with its (measurement) value stored in array vals_unit[]
//      In order to ensure consistency, changing the measurement value of a unit must (automatically) change the values of all others
// -5-  The unit names are stored in array names_unit[]
// -7-  All values concerning the units can only be populated, when a real converter ist instantiated.
// -8-  Additional methods deal with printing the conversion tables:
//      Each column represents a unit (the first column linked to the reference unit)
//      Each line   represents the unit values associated with the same state


// Registered converters (physical parameters) are identified (id, names)
// by means of enum_converter and ALLOWED_CONVERTER_NAMES
// -1- id of the supported converters
enum    enum_converter {
            conv_temperature, // id=0  -> Celsius, Fahrenheit, kelvin
            conv_time,        // id=1 -> sec, hour, day, year (=365.25 days)
//          conv_distance,    // id=2 -> km, miles
//          ...
            nALLOWED_CONVERTER_NAMES
        };
// -2- names of the supported converters
const   QString     ALLOWED_CONVERTER_NAMES[]  = {"temperature", "time", "distance"};

//-------------------------------------------------------------------------------------------------
class converter_generic
//-------------------------------------------------------------------------------------------------
{
//### VARIABLES
public:
   static int      n_Conv;                              // Number of converter objects generated so far.

protected:
//  VALUES POPULATED AT "THE GENERIC" LEVEL
    int             id_Conv;                            // Identification number of this converter object
    QString         name_converter;                     // Converter names is by default derived from id_Conv

//  VALUES POPULATED AT THE "PHYSICAL" LEVEL
    int             n_units;                             // Number of units register for a given converter
    QString*        names_unit;                          // ARRAY carrying the names of the units registered for this converter
    double*         vals_unit;                           // ARRAY bearing measurement values for each unit registered (the values stored should be consistent)
    int             id_ref_unit;                         // Which measurement unit is selected as the reference unit

//### FUNCTIONS

public:
// FUNCTIONS CONSTRUCTORS DESTRUCTORS
   converter_generic(                                  void);
   converter_generic(                                  QString convNam);
   virtual ~converter_generic(                         );

// INITIALIZATION AT GENERIC LEVEL
private:
    void            init(                               void);
// Initialization steps associated with the converter as such
    void            init_generic_id(                    void);
    void            init_generic_name(                  void);

// Initialization steps targeting at the units manipulated by this converter
// At the generic level one can only write that there is no known unit yet to be handled with
    void            init_generic_units(                 void);

public:
    void            setName_Converter(                  const QString convNam);
    QString         getName_Converter(                  void);

    int             get_nUnit(                          void);                          // get number of units associated

    void            set_UID_Ref(                        int uid);                       // set id of the reference unit
    int             get_UID_Ref(                        void);                          // get id of the reference unit
    void            set_UID_Ref_FromConsole(            void);                          // set id of the reference unit

    double          getVal_UID(                         const int uid);                 // get current measurement value associated with unit uid
    double          getVal_UID_Ref(                     void);                          // get current measurement value of the reference unit (has a known unit id)

    QString         getName_UID(                        const int uid);
    QString         getName_UID_Ref(                    void);
    void            prt_Units(                          void);


    virtual void    generateArray_names_unit(          void)= 0;
    virtual void    generateArray_vals_unit(            void)= 0;
    virtual void    setVal_UID(                         double val, int uid)=0;           // set current measurement value associated with unit uid
    void            setVal_UID_Ref(                     double val);                    // set current measurement value of the reference unit


//  GENERATING A LINE OF EQUIVALENT VALUES (DEPENDING OF THE UNIT USED)

private:
    QString         getName_ConvTabCol(                  const int col_no);
    double          getVal_ConvTabCol(                   const int col_no);                    // If the reference

public:
    void            wri_HeadLine_ConvTab(               void);
    void            wri_ValLine_ConvTab_4_UID_Ref(      double val);


};

#endif // CONVERTER_GENERIC_H
