/*
 * Project          C++ Implementation "Unit Conversion Table"
 * Author           Amar Khelil, www.khelil.de, Copyright 2014-2016
 * Documentation    http://www.khelil.de/Ref_Projekte_Intern/C/Project_C++_ConversionTable_EN.html
 *
 * FILE             converter_generic.cpp
 * Last modified    12 Feb 2016
 */

//C++ Standard Library
#include <iostream>
#include <iomanip>
#include <stddef.h>

//implemented in this project...
#include "global_UnitConversionTable.h"
#include "converter_generic.h"

/*
 * GENERIC converter is THE BASIC CLASS OF CONVERTERS (temperatur etc...)
 */

int converter_generic::n_Conv = 0;     // Number of converters generated: so far none.

//#################################################################################################
// SPECIFIC CONVERTER METHODS
//#################################################################################################

// CONSTRUCTORS OF A GENERIC CONVERTER OBJECT

//=====================================
//### CONSTRUCTORS DESTRUCTORS
//=====================================

//-------------------------------------------------------------------------------------------------
            converter_generic::converter_generic(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">Constructor converter_generic::converter_generic() [Begin]";
    }

    init();

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout   << std::endl << ">Constructor converter_generic::converter_generic() [End]";
    }
}   //  converter_generic::converter_generic(void)

//-------------------------------------------------------------------------------------------------
            converter_generic::converter_generic(QString convNam)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Constructor converter_generic::converter_generic" << "("<< convNam.toLocal8Bit().data()<< ") [Begin]";
    }

    init();
    setName_Converter(convNam); // The automatic name of converter is overwritten.

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {

        std::cout << std::endl
                  << ">Constructor converter_generic::converter_generic"<< "("<< convNam.toLocal8Bit().data()<< ") [End]";
    }

}   //  converter_generic::converter_generic(QString convNam)


//-------------------------------------------------------------------------------------------------
            converter_generic::~converter_generic()
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Destructor converter_generic::~converter_generic"
                  << "(name="<< name_converter.toLocal8Bit().data()
                  << ", id="<< id_Conv<<  ") [Begin]";
    }

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << "->Delete Array vals_unit[]  (unit values associated)";
    }
    delete[] vals_unit;

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << "->Delete Array names_unit[] (unit names associated)";
    }
    delete[] names_unit;

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl
                  << ">Destructor converter_generic::~converter_generic"
                  << "(name="<< name_converter.toLocal8Bit().data()
                  << ", id="<< id_Conv<<  ") [End]";
    }

}   //  converter_generic::converter_generic(QString convNam)

//=====================================
//### INITIALIZE OBJECT
//=====================================

//-------------------------------------------------------------------------------------------------
void        converter_generic::init(void)
//-------------------------------------------------------------------------------------------------
{
//  Variables at the generic level, basic initialization
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl<< ">converter_generic::init() [Begin]";
    }

    n_Conv++;                                                       // increment number of converter objects

    init_generic_id();
    init_generic_name();
    init_generic_units();

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl<< ">converter_generic::init() [End]";
    }
    return;
}   //  converter_generic::converter_generic(void)


//-------------------------------------------------------------------------------------------------
void   converter_generic::init_generic_id(                  void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl<< ">converter_generic::init_generic_id"<< "("<< id_Conv <<") [Begin]";
    }

    id_Conv             =  converter_generic::n_Conv;   // set identification number

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl<< ">converter_generic::init_generic_id"<< "("<< id_Conv <<") [End]";
    }
    return;
}   // void init_generic_name(void)

//-------------------------------------------------------------------------------------------------
void   converter_generic::init_generic_name(                  void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">converter_generic::init_generic_name(" << name_converter.toLocal8Bit().data() << ") [Begin]";
    }

    name_converter      =  "Generic_Converter_id_" + QString::number(id_Conv);  // set a generic name of converter

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">converter_generic::init_generic_name(" << name_converter.toLocal8Bit().data() << ") [End]";
    }
    return;
}   // void init_generic_name(void)


//-------------------------------------------------------------------------------------------------
 void   converter_generic::init_generic_units(void)
//-------------------------------------------------------------------------------------------------
{
     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
        std::cout << std::endl
                  << ">converter_generic::init_generic_units() [Begin] (a generic converter has no units)";
     }
/*
 * A generic converter has no units
 * Associated arrays have no address
 */
    n_units     = 0;        // set of units is empty
    names_unit  = NULL;     // pointer to null: no unit names...
    vals_unit   = NULL;     // pointer to null: no unit values
    id_ref_unit = 0;        // By default the first unit registered, this is the only value that is effective

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
       std::cout << std::endl << ">>converter_generic::init_generic_units() [End]";
    }
    return;
}

 //-------------------------------------------------------------------------------------------------
 void    converter_generic::setName_Converter( const QString convNam)
 //-------------------------------------------------------------------------------------------------
 {
     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl<< ">converter_generic::setName_Converter(" << name_converter.toLocal8Bit().data() << ") [Begin]";
     }

// Any newly created converter gets automatically a unique name that can be overwritten
     name_converter  = convNam;

     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl<< ">converter_generic::setName_Converter(" << name_converter.toLocal8Bit().data() << ") [End]";
     }
     return;
 }

//### Methods associated with the CONVERTER as such

 //-------------------------------------------------------------------------------------------------
 QString       converter_generic::getName_Converter( void)
 //-------------------------------------------------------------------------------------------------
 {
     return name_converter;
 }

 //-------------------------------------------------------------------------------------------------
 int         converter_generic::get_nUnit(                      void)
 //-------------------------------------------------------------------------------------------------
 {
     return n_units;
 }

 //-------------------------------------------------------------------------------------------------
 void  converter_generic::set_UID_Ref(                        int uid)
 //-------------------------------------------------------------------------------------------------
 {
     if ( (uid>=0) && (uid < n_units) )
     {
        id_ref_unit = uid;
     }
    return;
 }  //  void  converter_generic::set_UID_Ref(int uid)

 //-------------------------------------------------------------------------------------------------
 int  converter_generic::get_UID_Ref(                        void)
 //-------------------------------------------------------------------------------------------------
 {
    return id_ref_unit;
 }  // int  converter_generic::get_UID_Ref(void)

 //-------------------------------------------------------------------------------------------------
 void  converter_generic::set_UID_Ref_FromConsole(            void)
 //-------------------------------------------------------------------------------------------------
 {
     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::set_UID_Ref_FromConsole() [Begin]";
     }

     prt_Units();

     //  Select unit
         int     n;                      // no need to initialize
         bool    answered    =false;

         do {
             std::cout << std::endl << "->Enter id of unit that shall be the reference unit in conversion table: ";
             std::cin  >> n;

             if ( (n>=0) && (n< get_nUnit()))
             {
                 set_UID_Ref( n);

                 if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
                 {
                     std::cout << "->Reference unit (name=" << getName_UID_Ref().toLocal8Bit().data()
                               << ", id=" << get_UID_Ref() << ")"
                               << std::endl;
                 }
                 answered   = true;
             }
             else
             {
                 std::cout << "->unit id=" << n << " is illegal (should be >= 0 and < "<< get_nUnit() << ")! Please try again.";
             }

         } while (! answered);

     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::set_UID_Ref_FromConsole() [End]";
     }
     return;
 }  // void  converter_generic::set_UID_Ref_FromConsole(            void)


 // Methods writing or reading the state of converter
 //-------------------------------------------------------------------------------------------------
 double  converter_generic::getVal_UID(                          const int uid)
 //-------------------------------------------------------------------------------------------------
 {
    double theVal;

     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::getVal_UID("<< uid << ") [Begin]";
     }

     if ( !get_nUnit() )
     {   // Error: If the converter is generic and has no units ARBITRARY VALUE is returned.
         theVal=glb_UCT::ARBITRARY_VALUE;
     }
     else if ( (uid>=0) && (uid < get_nUnit()))
     {  // Normal case: converter has units and passed id is legal
        theVal = vals_unit[uid];
     }
     else
     {
        /* Error: If passed unit id is illegal, return value of the current reference unit */
        theVal=vals_unit[id_ref_unit];
     }

     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::getVal_UID("<< uid << ") [End]";
     }

     return theVal;

 } // double  converter_generic::getVal_UID(int UID)



 //-------------------------------------------------------------------------------------------------
 double  converter_generic::getVal_UID_Ref(                      void)
 //-------------------------------------------------------------------------------------------------
 {
        return getVal_UID(id_ref_unit) ;
 } //    double  converter_generic::getVal_UID_Ref(void)



 //-------------------------------------------------------------------------------------------------
 QString     converter_generic::getName_UID(                     const int uid)
 //-------------------------------------------------------------------------------------------------
 {
     QString theNam;

     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::getName_UID() [Begin]";
     }


     if ( !get_nUnit() )
     {   // If the converter object has no unit, UNKNOWN NAME is returned.
         theNam=glb_UCT::UNKNOWN_NAME;
     }
     else if ( (uid>=0) && (uid < get_nUnit()))
     {  // Expected case
        theNam=names_unit[uid];
     }
     else
     {
        // If passed uid is outside range, return value associated with the reference unit
        theNam=names_unit[id_ref_unit];
     }

     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::getName_UID() [End]";
     }

     return theNam;
 }

 //-------------------------------------------------------------------------------------------------
 QString     converter_generic::getName_UID_Ref(               void)
 //-------------------------------------------------------------------------------------------------
 {
     return getName_UID(id_ref_unit);
 }

 //-------------------------------------------------------------------------------------------------
 void        converter_generic::prt_Units(                 void)
 //-------------------------------------------------------------------------------------------------
 {
     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::prt_Units() [Begin]";
     }

     std::cout << std::endl << "->List of units asscociated with "<< getName_Converter().toLocal8Bit().data();
     std::cout << std::endl << "  "
               << std::setw(glb_UCT::SETW_INDEX)<< "uid"
               << std::setw(glb_UCT::SETW_NAME) << "Name"
               << std::endl;

     for (int ii=0; ii< get_nUnit() ; ii++)
     {
         std::cout << "  "
                   << std::setw(glb_UCT::SETW_INDEX) << ii
                   << std::setw(glb_UCT::SETW_NAME)  << getName_UID(ii).toLocal8Bit().data()
                   << std::endl;
     }
     std::cout << std::endl;

     if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
     {
         std::cout << std::endl << ">converter_generic::prt_Units() [End]";
     }

     return ;
 }

//-------------------------------------------------------------------------------------------------
void        converter_generic::setVal_UID_Ref(double val)
//-------------------------------------------------------------------------------------------------
{
    return setVal_UID(val, id_ref_unit);
}


//### TABULAR REPRESENTATION of the data in a converter object

//-------------------------------------------------------------------------------------------------
QString     converter_generic::getName_ConvTabCol(              const int col_no)
//-------------------------------------------------------------------------------------------------
{
    // first column is associated with the reference unit
    int uid= (col_no + id_ref_unit) % n_units;
    return (names_unit[uid]);
}

//-------------------------------------------------------------------------------------------------
double  converter_generic::getVal_ConvTabCol(                   const int col_no)
//-------------------------------------------------------------------------------------------------
{
    // first column is associated with the reference unit
    int uid= (col_no + id_ref_unit) % n_units;
    return (vals_unit[uid]);
}

//-------------------------------------------------------------------------------------------------
void        converter_generic::wri_HeadLine_ConvTab( void)
//-------------------------------------------------------------------------------------------------
{
//  The Reference unit name is always the first to be printed
//  The reference unit is the source unit for conversion into the other existing ones
    for (int ii=0; ii< get_nUnit(); ii++)
    {
         std::cout << std::setw(glb_UCT::SETW_NAME)
                   << getName_ConvTabCol(ii).toLocal8Bit().data();
    }
    return;
}

//-------------------------------------------------------------------------------------------------
void        converter_generic::wri_ValLine_ConvTab_4_UID_Ref(   double val)
//-------------------------------------------------------------------------------------------------
//  The Reference unit value is always the first to be printed
{
//  set the value of the reference unit (the one for which the user wishes a conversion)
    std::setiosflags(std::ios::right);
    std::cout.precision(2);
    std::cout.setf(std::ios::fixed);
    setVal_UID_Ref(val);
    for (int ii=0; ii< get_nUnit(); ii++)
    {
         std::cout << std::setw(glb_UCT::SETW_NAME)<< getVal_ConvTabCol(ii);
    }
    std::cout << std::flush;
    std::resetiosflags(std::ios::right);
    return;
}

