/*
 * Project          C++ Implementation of a Command Line Program converting physical units and creating corresponding conversion tables
 * Author           Amar Khelil, www.khelil.de, Copyright 2014-2015
 * Documentation    http://www.khelil.de/Ref_Projekte_Intern/C/Project_C++_ConversionTable_EN.html
 * 
 *  DECLARATION     of class cl_calculator, Blueprint of a Command Line Calculator
 *
 *  The calculator maintains a list of physical converters (ListOfConverters).
 *  Each one dedicated to a physical parameter that is associated with a set of registered units.
 */


#ifndef CL_CALCULATOR_H
#define CL_CALCULATOR_H

// C, C++ FRAMEWORK
#include <iomanip>

// QT FRAMEWORK
#include <QString>

// PROJECT SPECIFIC DECLARATIONS
#include <converter_generic.h>          // the basic converter class
#include <converter_temperature.h>      // derived converter associated with temperature
#include <converter_time.h>             // derived converter associated with time

// A cl_calculator object controls the whole unit conversion process.
// Here how it works:
// -1- Select a converter from the list ALLOWED_CONVERTER_NAMES[] associated with the ids in enum_converter
// -2- Select the reference unit associated to this converter
// -3- Enter the conversion table parameter
// -4- Start calculation and output

// Following behaviour is intended:
// -    THERE SHALL BE ONLY ONE CALCULATOR OBJECT (cl_calculator is a singleton class)
// -    The singleton cl_calculator object instantiates on demand the unit converter selected by the user
//      (if it has not already been instantiated)
//


// CLASS COMMAND LINE CALCULATOR
//-------------------------------------------------------------------------------------------------
class cl_calculator
//-------------------------------------------------------------------------------------------------
{

public:
//  Maintaining a list calculators (reduced to a single one)
    static int                      n_calc;

private:
    static  cl_calculator*          TheCalculator;

//  Identification number of this cl_calculator object 
    int                 calc_id;
    QString             calc_name;

//  cl_calculator maintain a static list of converters
//  Initially this list is empty: All elements of array are set to NULL pointers
//  Converters objects will be instantiated as needed.
    static converter_generic*       ListOfConverters[nALLOWED_CONVERTER_NAMES];

//  INPUT DATA required to perform a conversion 
//  Which physical parameter shall be processed - meaning which converter is currently selected
//  The currently selected converter is 
    int                 id_cur_converter ;      // identification of the currently selected converter
    QString             name_cur_converter;     // the name of the current converter ALLOWED_CONVERTER_NAMES[id_cur_converter]

//  Characteristics of the conversion table to generate 
    double              min_ValTab;             // Min value of conversion table
    double              max_ValTab;             // Max value of conversion table
//  Two different ways to enter the same piece of information from which the number of cconversion is derived
    double              delta_ValTab;           // Delta-Value of conversion table
    int                 n_ValTab;               // Number of values calculated

//  Each calculation of a conversion table invoking do_iteration() increments the number of iteration
    int                 n_iteration;

private:
//  Instantiation step:
    cl_calculator(                                               void);

public:
//  Instantiation step:
    static  cl_calculator*          getTheCalculator(            void);
    static int                      get_numberOf_cl_calculator(  void);
// post process step
    ~cl_calculator();

//  INITIALIZATION STEP 1 : Identification of calculator
    int     get_n_iteration(void){return n_iteration;}

    void    set_calc_id(                                         void);
    int     get_calc_id(                                         void);
    void    set_calc_name(                                       void);
    QString get_calc_name(                                       void);

//  INITIALIZATION STEP 2 : SELECTING/CREATING/CONFIGURING A CONVERTER
    void    prt_Converters(                                     void);

    void    set_id_SelectedConverter(                            const int idCurConv);
    int     get_id_SelectedConverter(                            void);
    void    prt_id_SelectedConverter(                            void);


    void    setNam_SelectedConverter(                            const QString NamCurConv);
    void    setNam_SelectedConverter(                            const int idCurConv);
    QString getNam_SelectedConverter(                            void);

    converter_generic* get_SelectedConverter(                    void);

    void    upd_ConverterList(                                   const int idCurConv);
    void    upd_ConverterList(                                   void);
    void    sel_Converter_FromConsole(                           void);

//  INITIALIZATION STEP 3 : CONFIGURING THE CONVERSION TABLE: min, max, number of values to calculate or their incrementation
//  -1- min, max, Number of values  (incrementation is calculated)
    void    set_ConversionTab_MinMax_n(                         double min, double max, unsigned int n=1);
    void    set_ConversionTab_MinMax_n_FromConsole(             void);
//  -2- min, max, incrementation value (number of values is calculated)
    void    set_ConversionTab_MinMax_Delt(                      double min, double max, double delt=1.0);
    void    set_ConversionTab_MinMax_Delt_FromConsole(          void);

    void    prt_Pars_ConversionTab_MinMax(                      void);


//  CALCULATING THE CONVERSION TABLE
    void    calc_conversion_table(                              const int idCurConv);
    void    calc_conversion_table(                              void);

//  One iteration process is: INIT 2, INIT 3, CALC
    void    do_iteration(                                       void);
};

#endif 
