/*
 * Project          C++ Implementation of a Command Line Program converting physical units and creating corresponding conversion tables
 * Author           Amar Khelil, www.khelil.de, Copyright 2014-2015
 * Documentation    http://www.khelil.de/Ref_Projekte_Intern/C/Project_C++_ConversionTable_EN.html
 *
 * DEFINITION       of class cl_calculator, Blueprint of a Command Line Calculator
 */

// Framework C, C++
#include <iostream>
#include <iomanip>
#include <cmath>
#include <stddef.h>

// Framework qt
//#include <QString>

// Framework CL_UnitConversionTable
#include "global_UnitConversionTable.h"
#include "cl_calculator.h"

// INITIALIZATION STEP 0
int                 cl_calculator::n_calc                                       = 0;        // No cl_calculator object yet.
cl_calculator*      cl_calculator::TheCalculator                                = NULL;
converter_generic*  cl_calculator::ListOfConverters[nALLOWED_CONVERTER_NAMES]   = {NULL,};   // No converter object associated yet.

//-------------------------------------------------------------------------------------------------
    cl_calculator::cl_calculator()
//-------------------------------------------------------------------------------------------------
{
        if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
        {
            std::cout << std::endl << ">Constructor cl_calculator() (singleton class) [Begin]";
        }

//      -1- IDENTIFICATION OF THE GENERATED COMMAND LINE CALCULATOR
        set_calc_id( );
        set_calc_name();

        // No conversion table has been calculated yet
        n_iteration = 0;

        if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
        {
            std::cout << std::endl << ">Constructor cl_calculator() [End]";
        }
} //    cl_calculator::cl_calculator()

 //-------------------------------------------------------------------------------------------------
     cl_calculator* cl_calculator::getTheCalculator()
 //-------------------------------------------------------------------------------------------------
 {
     std::cout << std::setiosflags(std::ios::left);
 //  std::cout.fill('.');
     if (!TheCalculator)
     {
         std::cout << std::endl << ">cl_calculator::getTheCalculator() creates a calculator object";
         TheCalculator = new cl_calculator();
     }
     else
     {
         std::cout << std::endl << ">cl_calculator::getTheCalculator() (only) retrieves the calculator singleton";
     }
     return TheCalculator;
 } // cl_calculator::getTheCalculator()


//-------------------------------------------------------------------------------------------------
int     cl_calculator::get_numberOf_cl_calculator(void)
//-------------------------------------------------------------------------------------------------
{   // this is a singleton class then shall be only one calculator
   return n_calc;
}

//-------------------------------------------------------------------------------------------------
    cl_calculator::~cl_calculator()
//-------------------------------------------------------------------------------------------------
{
        if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
        {
            std::cout << std::endl << ">Destructor ~cl_calculator() [Begin]";
        }

        for (int ii=0; ii<nALLOWED_CONVERTER_NAMES; ii++)
        {
            if (ListOfConverters[ii])
            {
                if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
                {
                    std::cout << std::endl << ">Converter(no="<< ii<< ", name="<< (ListOfConverters[ii]->getName_Converter()).toLocal8Bit().data()<< ") is deleted.";
                }
                delete ListOfConverters[ii];
            }
            else
            {
                if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
                {
                    std::cout << std::endl<< ">Converter (no="<< ii<< ", name="<< ALLOWED_CONVERTER_NAMES[ii].toLocal8Bit().data()<< ") was NOT instantiated (no need to delete)";
                }
            }
        }
        if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
        {
            std::cout << std::endl << ">Destructor ~cl_calculator() [End]" << std::endl;
        }
}   // cl_calculator::~cl_calculator()

//-------------------------------------------------------------------------------------------------
//  INITIALIZATION STEP 1
//-------------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
void     cl_calculator::set_calc_id(void)
//-------------------------------------------------------------------------------------------------
{
// The id of a calculator object is simply its ranking in the instantiation sequence.
   calc_id   =  ++n_calc;
   if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
   {
    std::cout << std::endl << ">cl_calculator::set_calc_id("<< get_calc_id() <<")";
   }
   return;
}   // void     cl_calculator::set_calc_id(void)

//-------------------------------------------------------------------------------------------------
int     cl_calculator::get_calc_id(void)
//-------------------------------------------------------------------------------------------------
{
   if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
   {
        std::cout << std::endl << ">cl_calculator::get_calc_id("<< calc_id <<")";
   }
   return calc_id;
} // int     cl_calculator::get_calc_id(void)

//-------------------------------------------------------------------------------------------------
void     cl_calculator::set_calc_name(void)
//-------------------------------------------------------------------------------------------------
{
// the name of a calculator object is directly derived from its id
   calc_name = "calculator_" + QString::number(get_calc_id());
   if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
   {
       std::cout << std::endl << ">cl_calculator::set_calc_name("<< get_calc_name().toLocal8Bit().data() << ")";
   }
   return;
} // void     cl_calculator::set_calc_name(void)

//-------------------------------------------------------------------------------------------------
QString    cl_calculator::get_calc_name(void)
//-------------------------------------------------------------------------------------------------
{
   if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
   {
       std::cout << std::endl << ">cl_calculator::get_calc_name("<< calc_name.toLocal8Bit().data() << ")";
   }
   return calc_name;
} // QString    cl_calculator::get_calc_name(void)


//-------------------------------------------------------------------------------------------------
void    cl_calculator::prt_Converters(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::prt_Converters() [Begin]";
    }

    std::cout << std::endl<< "->For "<< get_calc_name().toLocal8Bit().data()<< " available converters are (id, name):"<< std::endl;

    for (int i=0; i<nALLOWED_CONVERTER_NAMES; i++)
    {
        std::cout << "  id=" << std::setw(glb_UCT::SETW_INT)   << i
                  << " name="<< std::setw(glb_UCT::SETW_NAME)  << ALLOWED_CONVERTER_NAMES[i].toLocal8Bit().data()
                  << std::endl;
    }

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::prt_Converters() [End]";
    }

    return;
} // void    cl_calculator::prt_Converters(void)


//-------------------------------------------------------------------------------------------------
void     cl_calculator::set_id_SelectedConverter(const int idCurConv)
//-------------------------------------------------------------------------------------------------
{
    if ( (idCurConv >=0) && ( idCurConv < nALLOWED_CONVERTER_NAMES))
    {
        id_cur_converter =  idCurConv;
    }
    else
    {
        std::cout << std::endl << ">ERROR: cl_calculator::set_id_SelectedConverter("<<idCurConv<<"): Illegal converter id: Nothing changes";
    }

   return;
}

//-------------------------------------------------------------------------------------------------
int     cl_calculator::get_id_SelectedConverter(void)
//-------------------------------------------------------------------------------------------------
{
   return id_cur_converter;
}


//-------------------------------------------------------------------------------------------------
void    cl_calculator::prt_id_SelectedConverter(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::prt_id_SelectedConverter() [Begin]";
    }
    std::cout << std::endl << "->Calculator name: " << calc_name.toLocal8Bit().data();
    std::cout << std::endl << "->Converter name : " << name_cur_converter.toLocal8Bit().data();

//  std::cout << std::endl;
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::prt_id_SelectedConverter() [End]";
    }
    return;
} // void    cl_calculator::prt_id_SelectedConverter(void)

//-------------------------------------------------------------------------------------------------
void     cl_calculator::setNam_SelectedConverter( QString namCurConv)
//-------------------------------------------------------------------------------------------------
{
   name_cur_converter = namCurConv;
   return ;
}

//-------------------------------------------------------------------------------------------------
void     cl_calculator::setNam_SelectedConverter( const int idCurConv)
//-------------------------------------------------------------------------------------------------
{
    if ( (idCurConv >=0) && ( idCurConv < nALLOWED_CONVERTER_NAMES))
    {
        name_cur_converter  = ALLOWED_CONVERTER_NAMES[idCurConv];
    }
    else
    {
        std::cout << std::endl << ">ERROR: cl_calculator::setNam_SelectedConverter("<<idCurConv<<"): Illegal converter id: Nothing changes";
    }
   return ;
}


//-------------------------------------------------------------------------------------------------
QString     cl_calculator::getNam_SelectedConverter(void)
//-------------------------------------------------------------------------------------------------
{
   return name_cur_converter;
}

//-------------------------------------------------------------------------------------------------
converter_generic*     cl_calculator::get_SelectedConverter(void)
//-------------------------------------------------------------------------------------------------
{
    int id    = get_id_SelectedConverter();
    if (!ListOfConverters[id])
    {
        std::cout << std::endl << ">ERROR: cl_calculator::get_SelectedConverter("<<id<<"): Illegal converter id: NO CONVERTER HAS BEEN SELECTED YET";
    }

   return ListOfConverters[id];
}

//-------------------------------------------------------------------------------------------------
void cl_calculator::upd_ConverterList(const int idCurConv)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::upd_ConverterList("<<idCurConv<<") [Begin]";
    }

//  Only IMPLEMENTED converter shall be selected
    switch (idCurConv) {

        case conv_temperature: //
            id_cur_converter    = conv_temperature;
            if (!ListOfConverters[conv_temperature]) ListOfConverters[conv_temperature] = new converter_temperature(ALLOWED_CONVERTER_NAMES[conv_temperature]);
            break;

        case conv_time: //
            id_cur_converter    = conv_time;
            if (!ListOfConverters[conv_time]) ListOfConverters[conv_time] = new converter_time(ALLOWED_CONVERTER_NAMES[conv_time]);
            break;

        default: // illegal converter selected (not implemented yet)

            std::cout << std::endl << "->ERROR: converter id=" << idCurConv << " is not implemented!";
            id_cur_converter    = conv_temperature;
            std::cout << std::endl << "->       converter id="<< id_cur_converter
                      << ", name="<< (ALLOWED_CONVERTER_NAMES[id_cur_converter]).toLocal8Bit().data()
                      << ") is selected instead!";

            if (!ListOfConverters[conv_temperature]) ListOfConverters[conv_temperature] = new converter_temperature(ALLOWED_CONVERTER_NAMES[conv_temperature]);
            break;

    } // end of switch


    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::upd_ConverterList("<<idCurConv<<") [End]";
    }

    return;
}

//-------------------------------------------------------------------------------------------------
void cl_calculator::upd_ConverterList(void)
//-------------------------------------------------------------------------------------------------
{
    upd_ConverterList(id_cur_converter);
    return;
}

//-------------------------------------------------------------------------------------------------
void    cl_calculator::sel_Converter_FromConsole(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::sel_Converter_FromConsole() [Begin]";
    }

    prt_Converters();

//  Select the converter to use
    int     n;                      // no need to initialize
    bool    answered    =false;

    do {
        std::cout << std::endl << "->Enter id of converter to select: ";
        std::cin  >> n;

        if ( (n>=0) && (n< nALLOWED_CONVERTER_NAMES))
        {
            set_id_SelectedConverter(n);
            setNam_SelectedConverter(n);

            if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
            {
                std::cout << "->Selected converter (name=" << getNam_SelectedConverter().toLocal8Bit().data()
                          << ", id=" << get_id_SelectedConverter() << ")"
                          << std::endl;
            }

            answered   = true;
        }
        else
        {
            std::cout << "->id=" << n << " is illegal (should be >= 0 and < "<< nALLOWED_CONVERTER_NAMES<< ")! Please try again.";
        }

    } while (! answered);

//  -3- The list of converter objects maintained by cl_calculator may need to be updated
    upd_ConverterList();

//  -4- The reference unit must be defined!
    get_SelectedConverter()->set_UID_Ref_FromConsole();

    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::sel_Converter_FromConsole() [End]";
    }
    return;
}   // void    cl_calculator::sel_Converter_FromConsole(void)



//-------------------------------------------------------------------------------------------------
void    cl_calculator::set_ConversionTab_MinMax_n_FromConsole(void)
//-------------------------------------------------------------------------------------------------
{
    std::cout << std::endl;
    std::cout << ">Set range of conversion table (method 1) :";
    double  min;
    std::cout << std::endl;
    std::cout << "-> Enter MIN value (min)                : ";
    std::cin  >> min;
    double  max;
    std::cout << "-> Enter MAX value (max)                : ";
    std::cin  >> max;
    int  n;
    std::cout << "-> Enter required number of values (n)  : ";
    std::cin  >> n;
    std::cout << std::endl;

    set_ConversionTab_MinMax_n(min, max, n);

    return;
}

//-------------------------------------------------------------------------------------------------
void    cl_calculator::set_ConversionTab_MinMax_n(double min, double max, unsigned int n)
//-------------------------------------------------------------------------------------------------
{
    // Consistency check
    if (min > max)
    {
        double  temp=min;
        min = max;
        max = temp;
    }

    double delt;
    if (n==0) {
        delt = (max-min);
    }
    else {
        delt = (max-min+1)/n;
    }
    // if min=max or approximately
    if (delt < 0.01)   delt = 1.0;

    // consistent values are registered
    min_ValTab   = min;
    max_ValTab   = max;
    delta_ValTab = delt;
    n_ValTab     = static_cast<int> ((max-min)/delt + 1);
            
    return;
}


//-------------------------------------------------------------------------------------------------
void    cl_calculator::set_ConversionTab_MinMax_Delt_FromConsole(void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">set_ConversionTab_MinMax_Delt_FromConsole() [Begin]";
    }
    std::cout << std::endl << "->Set range of conversion table (Method 2) :";
    double  min;
    std::cout << std::endl;
    std::cout << "-> Enter MIN value (min)                : ";
    std::cin  >> min;
    double  max;
    std::cout << "-> Enter MAX value (max)                : ";
    std::cin  >> max;
    double  delt;
    std::cout << "-> Enter increment value (delt)         : ";
    std::cin  >> delt;
    std::cout << std::endl;

    set_ConversionTab_MinMax_Delt(min, max, delt);
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">set_ConversionTab_MinMax_Delt_FromConsole() [End]";
    }
    return;
}


//-------------------------------------------------------------------------------------------------
void    cl_calculator::set_ConversionTab_MinMax_Delt(double min, double max, double delt)
//-------------------------------------------------------------------------------------------------
{
    // Consistency check
    if (min > max)
    {
        double  temp=min;
        min = max;
        max = temp;
    }

    if (delt <  0)      delt = -delt;
    // delta shall not be small than one hundredth of unit!
    if (delt <  0.01)   delt = (fmax((max-min), 1.0));

    // consistent values are registered
    min_ValTab   = min;
    max_ValTab   = max;
    delta_ValTab = delt;

    // Number of value to calculated is derived from incremental step
    // Due to imprecisions in calculation nVal may be inferior by one to the real number
    n_ValTab      = static_cast<int> ((max-min)/delt + 1.5);

    return;
}

//-------------------------------------------------------------------------------------------------
void    cl_calculator::prt_Pars_ConversionTab_MinMax(void)
//-------------------------------------------------------------------------------------------------
{
    std::cout << std::endl
              << ">cl_calculator::prt_Pars_ConversionTab_MinMax("
              << calc_name.toLocal8Bit().data() << ")";

    std::cout << std::endl  << "-> MIN value in conversion table (min ): "      << std::setw(glb_UCT::SETW_VAL) << min_ValTab;
    std::cout << std::endl  << "-> MAX value in conversion table (max ): "      << std::setw(glb_UCT::SETW_VAL) << max_ValTab;
    std::cout << std::endl  << "-> Increment value               (delt): "      << std::setw(glb_UCT::SETW_VAL) << delta_ValTab;
    std::cout << std::endl  << "-> Number of values              (n   ): "      << std::setw(glb_UCT::SETW_VAL) << n_ValTab;
    std::cout << std::endl;
    return;
}





//-------------------------------------------------------------------------------------------------
void cl_calculator::calc_conversion_table(      void)
//-------------------------------------------------------------------------------------------------
{
    return calc_conversion_table(id_cur_converter);
}

//-------------------------------------------------------------------------------------------------
void cl_calculator::calc_conversion_table(const int idCurConv)
//-------------------------------------------------------------------------------------------------
{
    std::cout << std::endl;
    std::cout << std::endl
              << std::setw(glb_UCT::SETW_METHOD_NAME)
              << ">cl_calculator::calc_conversion_table"
              << " according to "
              << calc_name.toLocal8Bit().data()
              << std::endl;

    converter_generic*    TheConv    = ListOfConverters[idCurConv];   // code legibility: the converter to use
    if (TheConv == NULL)
    {
        std::cout << std::endl
                  << "->ERROR Converter id="
 //               << std::setw(glb_UCT::SETW_INT)
                  << idCurConv
                  << " is not instantiated yet !? (treatment aborted)";
        return;
    }

    std::cout << std::endl
              << "->Converter associated to conversion table is "
//            << std::setw(glb_UCT::SETW_NAME)
              << TheConv->getName_Converter().toLocal8Bit().data();

    std::cout << std::endl
              << "->"
 //           << std::setw(glb_UCT::SETW_INT)
              << TheConv->get_nUnit()
              << " units registered: "
              << std::endl;


    TheConv->prt_Units();

    std::cout << std::endl
              << "-> "
//            << std::setw(glb_UCT::SETW_NAME)
              << TheConv->getName_UID_Ref().toLocal8Bit().data()
              << " is reference unit";
    std::cout << std::endl;

    std::cout << std::endl
              << "-> min value: "
              << std::setw(glb_UCT::SETW_INT)
              << min_ValTab << " "
//            << std::setw(glb_UCT::SETW_NAME)
              << TheConv->getName_UID_Ref().toLocal8Bit().data();


    std::cout << std::endl
              << "-> max value: "
              << std::setw(glb_UCT::SETW_INT)
              << max_ValTab << " "
//            << std::setw(glb_UCT::SETW_NAME)
              << TheConv->getName_UID_Ref().toLocal8Bit().data();

    //-1- table names
    std::cout << std::endl << "--------------------------------------------------------------------------------";
    std::cout << std::setw(glb_UCT::SETW_INDEX)
              << std::endl
              << "No";
              TheConv->wri_HeadLine_ConvTab();
    std::cout << std::endl << "--------------------------------------------------------------------------------";

    //-2- table values
    int     kk  = 1;
    double  val = min_ValTab;
    do {
             std::cout << std::endl << std::setw(glb_UCT::SETW_INDEX) << kk++;
             TheConv->wri_ValLine_ConvTab_4_UID_Ref(val);
             val += delta_ValTab;
    } while (val <= max_ValTab);

    std::cout << std::endl << std::setw(glb_UCT::SETW_INDEX) << kk++;
    TheConv->wri_ValLine_ConvTab_4_UID_Ref(val);

    return;
}

//-------------------------------------------------------------------------------------------------
void cl_calculator::do_iteration(      void)
//-------------------------------------------------------------------------------------------------
{
    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
        std::cout << std::endl << ">cl_calculator::do_iteration() [Begin]";
    }

    n_iteration++;

//  STEP 1 : SELECT CONVERTER
//  STEP 2 : SELECT REFERENCE UNIT
    sel_Converter_FromConsole();

//  STEP 3 : CONFIGURE THE CONVERSION TABLE
//  min_ValTab      // Min value of conversion table
//  max_ValTab      // Max value of conversion table
//  There are 2 way to determine the number of values to convert
//  -1- to  determine the expected number of values to calculate
//  n_ValTab        // Number of values to calculate
//  -2- to set the incremental step between 2 consecutive values
//  delta_ValTab    // Delta-Value of conversion table

    // set_ConversionTab_MinMax_n_FromConsole();
    set_ConversionTab_MinMax_Delt_FromConsole(); // probably a more suitable procedure

//  STEP 4 :
    calc_conversion_table();


    if (glb_UCT::get_output_level() == OUTPUT_DEBUG)
    {
         std::cout << std::endl << ">cl_calculator::do_iteration() [End]";
    }

    return;
}


